<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly
if ( ! class_exists( 'MTS_Testimonials' ) ) :

class MTS_Testimonials {

	function __construct() {
		$this->includes();

        add_action( 'init', array( $this, 'init' ) );
		add_action( 'plugins_loaded', array( $this, 'load_textdomain' ) );
	}

	function init() {
		// Load class instances
		$this->shortcodes = new MTS_Testimonials_Shortcodes;

		$this->create_post_type();
    	$this->create_taxonomy();

		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_css' ), 100 );
		add_action( 'wp_footer', array( $this, 'enqueue_js' ), 11 );
		add_filter( 'wp_review_excluded_post_types', array( $this, 'wp_review_exclude_testimonials' ) );
	}

	function load_textdomain() {
		load_plugin_textdomain( 'mts-testimonials', false, MTS_TESTIMONIALS_FOLDER . '/languages' );
	}

	/**
	 * Adds testimonial to the database
	 *
	 * @param array $args
	 * @return void
	 */
	function add_testimonial( $args = array() ) {
		if ( !empty ( $args ) ) {
			// Required fields
			if ( empty( $args['testimonial'] ) )
				return;

			$settings = mts_testimonials_get_settings();
			$moderate_testimonials = (int) $settings['general']['moderate_testimonials'];

			$post_status = ( $moderate_testimonials == true ? 'draft' : 'publish' );

			$title = ( !empty ( $args['title'] ) ? $args['title'] : __('Testimonial', 'mts-testimonials') );

			$testimonial_args = array(
				'post_title'    => $title,
				'post_content'  => $args['testimonial'],
				'post_status'   => $post_status,
				'post_type' => 'mts_testimonials',
				'post_author'   => 1
			);

			$testimonial_id = wp_insert_post( $testimonial_args );

			//if ( isset ( $args['title'] ) )
				//add_post_meta( $testimonial_id, 'testimonial_title', $args['title'] );
			if ( isset ( $args['name'] ) )
				add_post_meta( $testimonial_id, 'testimonial_name', $args['name'] );
			if ( isset ( $args['position'] ) )
				add_post_meta( $testimonial_id, 'testimonial_position', $args['position'] );
			if ( isset ( $args['link'] ) )
				add_post_meta( $testimonial_id, 'testimonial_link', $args['link'] );
			if ( isset ( $args['email'] ) )
				add_post_meta( $testimonial_id, 'testimonial_email', $args['email'] );
			if ( isset ( $args['image'] ) )
				add_post_meta( $testimonial_id, 'testimonial_image', $args['image'] );
			if ( isset ( $args['rating'] ) )
				add_post_meta( $testimonial_id, 'testimonial_rating', $args['rating'] );

			// Assign to category if required
			if (!empty($settings['general']['submit_category'])) {
				wp_set_object_terms( $testimonial_id, $settings['general']['submit_category'], 'mts_testimonials_category', false );
			}

			return true;
		}
	}

	/**
     * Include classes
     */
    function includes() {
		require( MTS_TESTIMONIALS_INCLUDES_PATH . '/shortcodes.class.php' );
		require( MTS_TESTIMONIALS_INCLUDES_PATH . '/shortcode-generator.class.php' );
	}

	/**
     * Create post type
     */
    function create_post_type() {
		register_post_type( 'mts_testimonials', array(
             'labels' => array(
                 'name' => __( 'Testimonials', 'mts-testimonials' ),
                'singular_name' => __( 'Testimonials', 'mts-testimonials' )
            ),
            'public' => true,
            'publicly_queryable' => false,
            'exclude_from_search' => true,
            'has_archive' => false,
            'menu_icon' => 'dashicons-testimonial',
            'supports' => array(
                 'title', 'editor'
            ),
	        'taxonomies' => array('mts_testimonials_category')
        ) );
    }

	/**
     * Create testimonial
     */
	function create_taxonomy() {
	    register_taxonomy(
	        'mts_testimonials_category',
	        'mts_testimonials',
	        array(
	            'label' => __( 'Categories' ),
	            'public' => true,
	            'rewrite' => false,
	            'hierarchical' => true
	        )
	    );
	}

	function wp_review_exclude_testimonials($excluded) {
	  $excluded[] = 'mts_testimonials'; // Don't allow reviews on pages
	  return $excluded;
	}


	/**
	* Enqueues JavaScript and CSS files
	*/
	function enqueue_css() {
		wp_enqueue_style( 'mts-testimonials', MTS_TESTIMONIALS_URL . '/assets/css/testimonial.css' );
		wp_enqueue_style( 'mtswpt-owl-carousel', MTS_TESTIMONIALS_URL . '/assets/css/mtswpt.owl.carousel.css' );
	}

	function enqueue_js() {
		wp_enqueue_script( 'mtswpt-owl-carousel', MTS_TESTIMONIALS_URL . '/assets/js/owl.carousel.min.js', array( 'jquery' ), null, true );
		wp_enqueue_script( 'mts-testimonials', MTS_TESTIMONIALS_URL . '/assets/js/testimonial.js', array( 'jquery', 'mtswpt-owl-carousel' ), null, true );
	}



	// Install plugin
	function install() {
		if ( get_option( 'mts_testimonials' ) != 'installed' ) {
			update_option( 'mts_testimonials', 'installed' );
		}
	}

	function color_brightness($hex, $steps) {
	    // Steps should be between -255 and 255. Negative = darker, positive = lighter
	    $steps = max(-255, min(255, $steps));

	    // Normalize into a six character long hex string
	    $hex = str_replace('#', '', $hex);
	    if (strlen($hex) == 3) {
	        $hex = str_repeat(substr($hex,0,1), 2).str_repeat(substr($hex,1,1), 2).str_repeat(substr($hex,2,1), 2);
	    }

	    // Split into three parts: R, G and B
	    $color_parts = str_split($hex, 2);
	    $return = '#';

	    foreach ($color_parts as $color) {
	        $color   = hexdec($color); // Convert to decimal
	        $color   = max(0,min(255,$color + $steps)); // Adjust color
	        $return .= str_pad(dechex($color), 2, '0', STR_PAD_LEFT); // Make two char hex code
	    }

	    return $return;
	}

	function stars_input($name, $value, $id = null, $class='', $readonly = false) {
		if (empty($id))
			$id = 'mts-testimonial-input-'.rand(1000,9999);
		$max_stars = 5;
		$value = (int) $value;

		$output = '<div class="mts-testimonial-stars-input'.($readonly ? ' mts-testimonial-stars-readonly' : '').' '.$class.'" id="'.$id.'-wrap" data-current="'.$value.'">';

		if (!$readonly)
			$output .= '<input type="hidden" name="'.$name.'" value="'.$value.'" id="'.$id.'">';

		$output .= '<div class="mts-testimonial-star-input-bg">';
		for ($i = 0; $i < $max_stars; $i++) {
			$output .= '<i class="font-icons font-icons-star icon-bg" data-num="'.($i+1).'"></i>';
		}
		$output .= '</div>';
		$output .= '<div class="mts-testimonial-star-input-stars" style="width:'.($value*(100/$max_stars)).'%;">';
		for ($i = 0; $i < $max_stars; $i++) {
			$output .= '<i class="font-icons font-icons-star" data-num="'.$i.'"></i>';
		}
		$output .= '</div>';
		$output .= '</div>';
		return $output;
	}

	function layouts_dropdown($name, $selected, $id, $class='') {
		$output = '<select name="'.$name.'" id="'.$id.'" class="'.$class.'">';
		$options = $this->get_layouts();
		foreach ($options as $option_value => $option_label) {
			$output .= '<option value="'.$option_value.'" '.selected( $option_value, $selected, false ).'>'.$option_label.'</option>';
		}
		$output .= '</select>';
		return $output;
	}

	function get_layouts() {
		return apply_filters( 'mts_testimonials_layouts',
			array(
				'bubble_grid' => __('Bubble Grid', 'mts-testimonials'),
				'bubble_grid_author_above' => __('Bubble Grid with author above and content below', 'mts-testimonials'),
				'card_box_grid' => __('Card Box Grid', 'mts-testimonials'),
				'card_box_grid_center_align' => __('Card Box Grid Center Aligned', 'mts-testimonials'),
				'card_box_grid_with_author_content' => __('Card Box Grid with author below and content above', 'mts-testimonials'),
				'card_box_grid_with_author_above_content' => __('Card Box Grid with author above and content below', 'mts-testimonials'),
				'card_box_full_width' => __('Card Box Full Width', 'mts-testimonials'),
				'quote_grid' => __('Quote Grid', 'mts-testimonials'),
				'quote_grid_center_align' => __('Quote Grid Center Aligned', 'mts-testimonials'),
				'quote_grid_with_author_above' => __('Quote Grid with author above and content below', 'mts-testimonials'),
				'quote_full_width' => __('Quote Full Width', 'mts-testimonials'),
				'simple_grid' => __('Simple Grid', 'mts-testimonials'),
				'carousel_testimonial_grid' => __('Carousel Testimonial Grid', 'mts-testimonials'),
				'carousel_testimonial' => __('Carousel Testimonial Full Width', 'mts-testimonials'),
			)
		);
	}
}

endif;
